%
% t-CWT Copyright (C) 2003-2015 Vladimir Bostanov. Type "tcwt_help".
%

% This file is part of t-CWT.
%
% t-CWT is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published
% by the Free Software Foundation, either version 3 of the License,
% or (at your option) any later version.
%
% t-CWT is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty
% of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
% See the GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with t-CWT; see the file COPYING or the file tcwt_gpl.m
% If not, see <http://www.gnu.org/licenses/>.

%: tcwt_prm2info(prmFILE) displays the following values
%: computed from the input parameter file prmFILE <tcwt_help prm>:
%:
%:    T       Length of the time window ( = Window(2) - Window(1) )
%:    S       Cut-off scale ( = cutoffScale )
%:    R       Log-grid sampling rate ( = logGridR )
%:
%:    NcwMax  Maximum number of trials processed by CWT in one step
%:
%:    Nc      Number of channels to be processed by t-CWT
%:    Nf      Number of frequency components
%:    Nv      Number of log-grid vertexes
%:
%:    Npca  = 2*(Nc*Nf)^2  Number of PCT and covariance matrix elements
%:
%:    Ncwt  = Nc*Nf*Nv     Exact number of non-zero CWT matrix elements
%:
%:    NcwtA = 12*Nc*((R*T)/S)^2   Approximate number of non-zero CWT ME
%:
%:    Tpca    Estimated processing time for one PCA iteration
%:            assuming 1 microsecond per PCA matrix element
%:
%:    Tcwt    Estimated processing time for one CWT scalogram
%:            assuming 1 microsecond per CWT matrix element
%:
%:    Mpca    Estimated minimum amount of memory needed for PCA
%:
%:    Mcwt    Estimated maximum amount of memory needed for CWT
%:
%: TIP: Try different parameter settings using the test parameter file
%:      tcwtROOT/jobs/test_prm.m' and typing "tcwt_prm2info test_prm".

function tcwt_prm2info(prmfile);

eval(prmfile);

Time = tcwt_erptime(NtimePt,a2dR,Epoch,Window,BaseLine);

N = length(Time);
Ti = Time(1);
Tf = Time(N) + Time(2) - Time(1);
Tw = Tf-Ti;

Nfreq = Tw/(cutoffScale/2);
maxFreq = floor(Nfreq);
if maxFreq>floor((N-1)/2)
 error('The value of the job input parameter cutoffScale is too small.')
end

minScale=N/ceil(Nfreq);
maxScale=4*N;

logGrid = tcwt_loggrid(N+1,minScale,maxScale,logGridR);

Nc = sum(ChannelList(:,size(ChannelList,2))==' ');
Nf = 1 + 2*maxFreq;
Nv = size(logGrid,2);

Npca = 2*(Nc*Nf)^2;
Ncwt = Nc*Nf*Nv;

Mpca = ceil(8e-6*Npca);
Mcwt = ceil(8e-6*Nv*(NcwMax+Nf+5));

NcwtA = round(12*Nc*((logGridR*Tw)/cutoffScale)^2);

tcwt_txt('\n');
tcwt_txt('~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n');
tcwt_txt('tcwt_prm2info: input parameter file: %s.m\n', prmfile);
tcwt_txt('---------------------------------------------------------------------------\n');
tcwt_txt('Length of the time window:  T = %5.3f seconds\n', Tw);
tcwt_txt('Cut-off scale :             S = %5.3f seconds\n', cutoffScale);
tcwt_txt('Log-grid sampling rate:     R = %u\n\n', logGridR);
tcwt_txt('Maximum number ot trials\n');
tcwt_txt('processed by CWT in one step:  NcwMax = %u\n\n', NcwMax);
tcwt_txt('Number of channels to be processed:  Nc = %u\n', Nc);
tcwt_txt('Number of frequency components:      Nf = %u\n', Nf);
tcwt_txt('Number of log-grid vertexes:         Nv = %u\n\n', Nv);
tcwt_txt('Number of matrix elements (ME):\n');
tcwt_txt('Number of PCA ME (PCT + Covariance):   Npca = 2*(Nc*Nf)^2 = %u\n', Npca);
tcwt_txt('Exact number of non-zero CWT ME:       Ncwt = Nc*Nf*Nv    = %u\n', Ncwt);
tcwt_txt('Approximate number of non-zero CWT ME: 12*Nc*((R*T)/S)^2  = %u\n\n', NcwtA);
tcwt_txt('Assuming 1 microsecond processing time per ME:\n');
tcwt_txt('Estimated processing time for one PCA iteration: Tpca = %f seconds\n', Npca*1e-6);
tcwt_txt('Estimated processing time for one CWT scalogram: Tcwt = %f seconds\n\n', Ncwt*1e-6);
tcwt_txt('Estimated minimum amount of memory needed for PCA: Mpca = %u MB\n\n', Mpca);
tcwt_txt('Estimated maximum amount of memory needed for CWT: Mcwt = %u MB\n', Mcwt);
tcwt_txt('___________________________________________________________________________\n');
